/**
 * Search Engine Keyword Highlight (http://fucoder.com/code/se-hilite/)
 *
 * This module can be imported by any HTML page, and it would analyse the
 * referrer for search engine keywords, and then highlight those keywords on
 * the page, by wrapping them around <span class="hilite">...</span> tags.
 * Document can then define styles else where to provide visual feedbacks.
 *
 * Usage:
 *
 *   In HTML. Add the following line towards the end of the document.
 *
 *     <script type="text/javascript" src="se_hilite.js"></script>
 *
 *   In CSS, define the following style:
 *
 *     .hilite { background-color: #ff0; }
 *
 *   If Hilite.style_name_suffix is true, then define the follow styles:
 *
 *     .hilite1 { background-color: #ff0; }
 *     .hilite2 { background-color: #f0f; }
 *     .hilite3 { background-color: #0ff; }
 *     .hilite4 ...
 *
 * @author Scott Yang <http://scott.yang.id.au/>
 * @version 1.5
 */

// Configuration:
Hilite = {
    /**
     * Element ID to be highlighted. If set, then only content inside this DOM
     * element will be highlighted, otherwise everything inside document.body
     * will be searched.
     */
    elementid: '',

    /**
     * Whether we are matching an exact word. For example, searching for
     * "highlight" will only match "highlight" but not "highlighting" if exact
     * is set to true.
     */
    exact: top.wholeWords!=null?top.wholeWords:false,

    /**
     * Maximum number of DOM nodes to test, before handing the control back to
     * the GUI thread. This prevents locking up the UI when parsing and
     * replacing inside a large document.
     */
    max_nodes: 500,

    /**
     * Whether to automatically hilite a section of the HTML document, by
     * binding the "Hilite.hilite()" to window.onload() event. If this
     * attribute is set to false, you can still manually trigger the hilite by
     * calling Hilite.hilite() in Javascript after document has been fully
     * loaded.
     */
    onload: true,

    /**
     * Name of the style to be used. Default to 'hilite'.
     */
    style_name: 'hilite',

    /**
     * Whether to use different style names for different search keywords by
     * appending a number starting from 1, i.e. hilite1, hilite2, etc.
     */
    style_name_suffix: false,

    /**
     * Set it to override the document.referrer string. Used for debugging
     * only.
     */
    debug_referrer: ''
};

Hilite.search_engines = [
    ['google\\.', 'q'],                             // Google
    ['search\\.yahoo\\.', 'p'],                     // Yahoo
    ['search\\.msn\\.', 'q'],                       // MSN
    ['search\\.live\\.', 'query'],                  // MSN Live
    ['search\\.aol\\.', 'userQuery'],               // AOL
    ['ask\\.com', 'q'],                             // Ask.com
    ['altavista\\.', 'q'],                          // AltaVista
    ['feedster\\.', 'q'],                           // Feedster
    ['search\\.lycos\\.', 'q'],                     // Lycos
    ['alltheweb\\.', 'q'],                          // AllTheWeb
    ['technorati\\.com/search/([^\\?/]+)', 1],      // Technorati
    ['dogpile\\.com/info\\.dogpl/search/web/([^\\?/]+)', 1, true] // DogPile
];

/**
 * Decode the referrer string and return a list of search keywords.
 */
Hilite.decodeReferrer = function(referrer) {
    var regex = new RegExp('');

	if (top.searchWordList != null && top.searchWordList.length > 0) {
		var query = new Array();
		query[0] = top.searchWordList[0];
		for (var i = 0; i<top.searchWordList.length;i++) {
			query[query.length]=top.searchWordList[i]
			if (i!=0) {
				query[0]+="[^\\w]+"+top.searchWordList[i];
			}
		}
		return query;
	} else return null;
	//modified to disable hilite from search engines
/*
    for (var i = 0; i < Hilite.search_engines.length; i ++) {
        var se = Hilite.search_engines[i];
        regex.compile('^http://(www\\.)?' + se[0], 'i');
        var match = referrer.match(regex);
        if (match) {
            var result;
            if (isNaN(se[1])) {
                result = Hilite.decodeReferrerQS(referrer, se[1]);
            } else {
                result = match[se[1] + 1];
            }
            if (result) {
                result = decodeURIComponent(result);
                // XXX: DogPile's URI requires decoding twice.
                if (se.length > 2 && se[2])
                    result = decodeURIComponent(result);
                result = result.replace(/\'|"/g, '');
                result = result.split(/[\s,\+\.]+/);
                return result;
            }
            break;
        }
    }
    return null;
*/
};

Hilite.decodeReferrerQS = function(referrer, match) {
    var idx = referrer.indexOf('?');
    var idx2;
    if (idx >= 0) {
        var qs = new String(referrer.substring(idx + 1));
        idx  = 0;
        idx2 = 0;
        while ((idx >= 0) && ((idx2 = qs.indexOf('=', idx)) >= 0)) {
            var key, val;
            key = qs.substring(idx, idx2);
            idx = qs.indexOf('&', idx2) + 1;
            if (key == match) {
                if (idx <= 0) {
                    return qs.substring(idx2+1);
                } else {
                    return qs.substring(idx2+1, idx - 1);
                }
            }
        }
    }
    return null;
};

/**
 * Highlight a DOM element with a list of keywords.
 */
Hilite.hiliteElement = function(elm, query) {
	if (!query || elm.childNodes.length == 0)
	return;

	var caseSens = top.caseSensitive;
	var qre = new Array();
    for (var i = 0; i < query.length; i ++) {
        query[i] = caseSens?query[i]:query[i].toLowerCase();
        if (Hilite.exact)
            qre.push('\\b'+query[i]+'\\b');
        else
            qre.push(query[i]);
    }

	for (var i = 0; i < qre.length; i ++) {
		qre[i] = new RegExp(qre[i], caseSens?"":"i")
	}
//	qre = new RegExp(qre.join("|"), caseSens?"":"i");

	var stylemapper = {};
    for (var i = 0; i < query.length; i ++) {
		stylemapper[0] = Hilite.style_name;
		if (Hilite.style_name_suffix) {
			if (i==0) {
				stylemapper[query[i].toLowerCase()] = Hilite.style_name+"1";
			} else {
				stylemapper[query[i].toLowerCase()] = Hilite.style_name;
			}
		} else {
			stylemapper[query[i].toLowerCase()] = Hilite.style_name;
		}
    }

	var textproc_simple = function(node) {
		for (var i=0;i<qre.length;i++) {
			match=qre[i].exec(node.data)
			if (match) {
				var val = match[0];
				var k = '';
				var node2 = node.splitText(match.index);
				var node3 = node2.splitText(val.length);
				var span = node.ownerDocument.createElement('SPAN');
				node.parentNode.replaceChild(span, node2);
				span.className = stylemapper[0];
				span.appendChild(node2);
				return span;
			}
		}
		return node;
	}

	var textproc = function(node,querryNum) {
		var q = querryNum;
		var match = null;
		var continuedSearch = true;
		if (querryNum == 0) {
			match=qre[q].exec(node.data);
			q++;
			querryNum++;
			continuedSearch = false;
		}
		if (match) {
            var val = match[0];
            var k = '';
            var node2 = node.splitText(match.index);
            var node3 = node2.splitText(val.length);
            var span = node.ownerDocument.createElement('SPAN');
			node.parentNode.replaceChild(span, node2);
            span.className = stylemapper[0];
            span.appendChild(node2);
            return span;
        } else {
			if (qre[q] && qre[q].exec(node.data)) {
				var words = normalizeSpace(node.data).split(/[\[ | \^ | \$ | \. | \| | \+ | \( | \) | ` | ~ | ! | # | % | & | - | \- | _ | = | \] | { | } | ; | ' | " | < | > | ,]/);
				var w = 0;
				var wrongSearch = false;
				if (q==1) {
					for (w = 0;w<words.length;w++) {
						if (qre[q].exec(words[w])) {
							break;
						}
					}
				}
				while (q<qre.length && w<words.length) {
					if (qre[q].exec(words[w])) {
						q++;
						w++;
					} else if (top.StopWords[words[w].toLowerCase()] || words[w].length == 0) {
						w++;
					} else {
						wrongSearch = true;
						break;
					}
				}
				if (!wrongSearch) {
					q--;
					if (q+1 == qre.length) {
						var matchS = qre[querryNum].exec(node.data);
						var node2 = node.splitText(matchS.index);
						var matchE = qre[q].exec(node2.data);
						var node3 = node2.splitText(matchE.index+matchE[0].length);
						var span = node.ownerDocument.createElement('SPAN');
						node.parentNode.replaceChild(span, node2);
						span.className = stylemapper[0];
						span.appendChild(node2);
						return span;
					} else {
						var nextNode = Hilite.getFollowingTextNode(node);
						if (nextNode != null) {
							var nextProcessedNode = textproc(nextNode,q+1)
							if (nextNode != nextProcessedNode) {
								var matchS = qre[querryNum].exec(node.data);
								var node2 = node.splitText(matchS.index);
								var span = node.ownerDocument.createElement('SPAN');
								node.parentNode.replaceChild(span, node2);
								span.className = stylemapper[0];
								span.appendChild(node2);
								return nextProcessedNode;
							}
						}
					}
				} else if (!continuedSearch && w<words.length) {
					var matchT = (new RegExp(words[w],"i")).exec(node.data)
					if (matchT) {
						var node2 = node.splitText(matchT.index+matchT[0].length);
						if (node.data.length-matchT.index-matchT[0].length == 0) {
							return textproc_simple(node);
						} else {
							return textproc(node2,0);
						}
					}
				}
			}
			return textproc_simple(node);
		}
/*
		var match = qre.exec(node.data);
        if (match) {
            var val = match[0];
            var k = '';
            var node2 = node.splitText(match.index);
            var node3 = node2.splitText(val.length);
            var span = node.ownerDocument.createElement('SPAN');
			node.parentNode.replaceChild(span, node2);
            span.className = stylemapper[0];
            span.appendChild(node2);
            return span;
        } else {
            return node;
        }
*/
    };
	Hilite.walkElements(elm.childNodes[0], 1, textproc);
};

Hilite.disHilite = function(node) {
	if (node != null) {
		if (node.nodeType == 1 || node.nodeType == 9) { // ELEMENT_NODE
			if ((node.nodeName.toLowerCase()=="span") && (node.attributes["class"] != null) && (node.attributes["class"].nodeValue.indexOf(Hilite.style_name) == 0)) {
				var childs = node.childNodes;
				for (var j=0;j<childs.length;j++) {
					node.parentNode.insertBefore(childs[j],node);
				}
				node.parentNode.removeChild(node);
			} else if (node.childNodes.length > 0) {
				var childs = node.childNodes;
				for (var i=0;i<childs.length;i++) {
					Hilite.disHilite(childs[i]);
				}
			}
		}
	}
}

/**
 * Highlight a HTML document using keywords extracted from document.referrer.
 * This is the main function to be called to perform search engine highlight
 * on a document.
 *
 * Currently it would check for DOM element 'content', element 'container' and
 * then document.body in that order, so it only highlights appropriate section
 * on WordPress and Movable Type pages.
 */
Hilite.hilite = function() {
    // If 'debug_referrer' then we will use that as our referrer string
    // instead.
	//BS: added Highlitinh switches
	if ((top.highlightingEnabled) && (top.highlightingShow)) {
		var q = Hilite.debug_referrer ? Hilite.debug_referrer : document.referrer;
		var e = Hilite.elementid?document.getElementById(Hilite.elementid):document.body;
		q = Hilite.decodeReferrer(q);
		if (q && (/*(Hilite.elementid &&
				   (e = document.getElementById(Hilite.elementid))) ||*/
				  (e != null)))
		{
		Hilite.hiliteElement(e, q);
		}
		top.highlightingShow = false;
	} else if (!top.highlightingEnabled) {
		Hilite.disHilite(document.body);
	}
};

Hilite.walkElements = function(node, depth, textproc) {
    var skipre = /^(script|style|textarea)/i;
    var count = 0;
    while (node && depth > 0) {
        count ++;
        if (count >= Hilite.max_nodes) {
            var handler = function() {
                Hilite.walkElements(node, depth, textproc);
            };
            setTimeout(handler, 50);
            return;
        }

        if (node.nodeType == 1) { // ELEMENT_NODE
            if (!skipre.test(node.tagName) && node.childNodes.length > 0) {
                node = node.childNodes[0];
                depth ++;
                continue;
            }
        } else if (node.nodeType == 3) { // TEXT_NODE
            node = textproc(node,0);
			if (node.parentNode == null) {
				alert(node.nodeName+":1"+node.data+":");
			}
		}

		if (node==null) {
			return;
		}

		if (node.nextSibling) {
            node = node.nextSibling;
        } else {
            while (depth > 0) {
                node = node.parentNode;
                depth --;
				if (node.nextSibling) {
                    node = node.nextSibling;
                    break;
                }
            }
        }
    }
};

Hilite.getFollowingTextNode = function(node) {
	node = Hilite.getFollowingNode(node);
	if (node) {
		if (node.nodeType == 3 && normalizeSpace(node.data).length > 0) {
			return node;
		} else {
			return Hilite.getFollowingTextNode(node);
		}
	} else {
		return null;
	}
}

Hilite.getFollowingNode = function(node) {
	if (node) {
		if (node.firstChild) {
			return node.firstChild;
		} else if (node.nextSibling) {
			return node.nextSibling;
		} else {
			while (node.parentNode) {
				node = node.parentNode;
				if (node.nextSibling) {
					return node.nextSibling;
				}
			}
			return null;
		}
	} else return null;
}

function normalizeSpace(string)
{
	var regexp = new RegExp("[\\s][\\s]+","g");
    while(string.match(regexp)) {
		string = string.replace(regexp," ");
	}
	if (string.length == 1 && string == " ") {
		return "";
	} else {
		return string;
	}
}


// Trigger the highlight using the onload handler.
if (Hilite.onload) {
    if (window.attachEvent) {
        window.attachEvent('onload', Hilite.hilite);
    } else if (window.addEventListener) {
        window.addEventListener('load', Hilite.hilite, false);
    } else {
        var __onload = window.onload;
        window.onload = function() {
            Hilite.hilite();
            __onload();
        };
    }
}

// SIG // Begin signature block
// SIG // MIIVVAYJKoZIhvcNAQcCoIIVRTCCFUECAQExCzAJBgUr
// SIG // DgMCGgUAMGcGCisGAQQBgjcCAQSgWTBXMDIGCisGAQQB
// SIG // gjcCAR4wJAIBAQQQEODJBs441BGiowAQS9NQkAIBAAIB
// SIG // AAIBAAIBAAIBADAhMAkGBSsOAwIaBQAEFLFq09ECnya4
// SIG // n/T4Y1ieQVCT2kYHoIIRFjCCA3owggJioAMCAQICEDgl
// SIG // 1/r4Ya+e9JDnJrXWWtUwDQYJKoZIhvcNAQEFBQAwUzEL
// SIG // MAkGA1UEBhMCVVMxFzAVBgNVBAoTDlZlcmlTaWduLCBJ
// SIG // bmMuMSswKQYDVQQDEyJWZXJpU2lnbiBUaW1lIFN0YW1w
// SIG // aW5nIFNlcnZpY2VzIENBMB4XDTA3MDYxNTAwMDAwMFoX
// SIG // DTEyMDYxNDIzNTk1OVowXDELMAkGA1UEBhMCVVMxFzAV
// SIG // BgNVBAoTDlZlcmlTaWduLCBJbmMuMTQwMgYDVQQDEytW
// SIG // ZXJpU2lnbiBUaW1lIFN0YW1waW5nIFNlcnZpY2VzIFNp
// SIG // Z25lciAtIEcyMIGfMA0GCSqGSIb3DQEBAQUAA4GNADCB
// SIG // iQKBgQDEtfJSFbyIhmApFkpbL0uRa4eR8zVUWDXq0TZe
// SIG // Yk1SUTRxwntmHYnI3SrEagr2N9mYdJH2kq6wtXaW8alK
// SIG // Y0VHLmsLkk5LK4zuWEqL1AfkGiz4gqpY2c1C8y3Add6N
// SIG // q8eOHZpsTAiVHt7b72fhcsJJwp5gPOHivhajY3hpFHut
// SIG // LQIDAQABo4HEMIHBMDQGCCsGAQUFBwEBBCgwJjAkBggr
// SIG // BgEFBQcwAYYYaHR0cDovL29jc3AudmVyaXNpZ24uY29t
// SIG // MAwGA1UdEwEB/wQCMAAwMwYDVR0fBCwwKjAooCagJIYi
// SIG // aHR0cDovL2NybC52ZXJpc2lnbi5jb20vdHNzLWNhLmNy
// SIG // bDAWBgNVHSUBAf8EDDAKBggrBgEFBQcDCDAOBgNVHQ8B
// SIG // Af8EBAMCBsAwHgYDVR0RBBcwFaQTMBExDzANBgNVBAMT
// SIG // BlRTQTEtMjANBgkqhkiG9w0BAQUFAAOCAQEAUMVLyCSA
// SIG // 3+QNJMLeGrGhAqGmgi0MgxWBNwqCDiywWhdhtdgF/ojb
// SIG // 8ZGRs1YaQKbrkr44ObB1NnQ6mE/kN7qZicqVQh2wuceg
// SIG // jVfg+tVkBEI1TgHRM6IXyE2qJ8fy4YZMAjhNg3jG/FPg
// SIG // 6+AGh92klp5eDJjipb6/goXDYOHfrSjYx6VLZNrHG1u9
// SIG // rDkI1TgioTOLL4qa67wHIT9EQQkHtWUcJLxI00SA66HP
// SIG // yQK0FM9UxxajgFz5eT5dcn2IF54sQ6LKU859PfYqOrhP
// SIG // lAClbQqDXfleU/QYs1cPcMP79a2VoA4X3sQWgGDJDytu
// SIG // hgTx6/R4J9EFxe40W165STLyMzCCA8QwggMtoAMCAQIC
// SIG // EEe/GZXfjVJGQ/fbbUgNMaQwDQYJKoZIhvcNAQEFBQAw
// SIG // gYsxCzAJBgNVBAYTAlpBMRUwEwYDVQQIEwxXZXN0ZXJu
// SIG // IENhcGUxFDASBgNVBAcTC0R1cmJhbnZpbGxlMQ8wDQYD
// SIG // VQQKEwZUaGF3dGUxHTAbBgNVBAsTFFRoYXd0ZSBDZXJ0
// SIG // aWZpY2F0aW9uMR8wHQYDVQQDExZUaGF3dGUgVGltZXN0
// SIG // YW1waW5nIENBMB4XDTAzMTIwNDAwMDAwMFoXDTEzMTIw
// SIG // MzIzNTk1OVowUzELMAkGA1UEBhMCVVMxFzAVBgNVBAoT
// SIG // DlZlcmlTaWduLCBJbmMuMSswKQYDVQQDEyJWZXJpU2ln
// SIG // biBUaW1lIFN0YW1waW5nIFNlcnZpY2VzIENBMIIBIjAN
// SIG // BgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEAqcqypMzN
// SIG // IK8KfYmsh3XwtE7x38EPv2dhvaNkHNq7+cozq4QwiVh+
// SIG // jNtr3TaeD7/R7Hjyd6Z+bzy/k68Numj0bJTKvVItq0g9
// SIG // 9bbVXV8bAp/6L2sepPejmqYayALhf0xS4w5g7EAcfrkN
// SIG // 3j/HtN+HvV96ajEuA5mBE6hHIM4xcw1XLc14NDOVEpkS
// SIG // ud5oL6rm48KKjCrDiyGHZr2DWFdvdb88qiaHXcoQFTyf
// SIG // hOpUwQpuxP7FSt25BxGXInzbPifRHnjsnzHJ8eYiGdvE
// SIG // s0dDmhpfoB6Q5F717nzxfatiAY/1TQve0CJWqJXNroh2
// SIG // ru66DfPkTdmg+2igrhQ7s4fBuwIDAQABo4HbMIHYMDQG
// SIG // CCsGAQUFBwEBBCgwJjAkBggrBgEFBQcwAYYYaHR0cDov
// SIG // L29jc3AudmVyaXNpZ24uY29tMBIGA1UdEwEB/wQIMAYB
// SIG // Af8CAQAwQQYDVR0fBDowODA2oDSgMoYwaHR0cDovL2Ny
// SIG // bC52ZXJpc2lnbi5jb20vVGhhd3RlVGltZXN0YW1waW5n
// SIG // Q0EuY3JsMBMGA1UdJQQMMAoGCCsGAQUFBwMIMA4GA1Ud
// SIG // DwEB/wQEAwIBBjAkBgNVHREEHTAbpBkwFzEVMBMGA1UE
// SIG // AxMMVFNBMjA0OC0xLTUzMA0GCSqGSIb3DQEBBQUAA4GB
// SIG // AEpr+epYwkQcMYl5mSuWv4KsAdYcTM2wilhu3wgpo17I
// SIG // ypMT5wRSDe9HJy8AOLDkyZNOmtQiYhX3PzchT3AxgPGL
// SIG // OIez6OiXAP7PVZZOJNKpJ056rrdhQfMqzufJ2V7duyuF
// SIG // PrWdtdnhV/++tMV+9c8MnvCX/ivTO1IbGzgn9z9KMIIE
// SIG // vzCCBCigAwIBAgIQQZGhWjl4389JZWY4HUx1wjANBgkq
// SIG // hkiG9w0BAQUFADBfMQswCQYDVQQGEwJVUzEXMBUGA1UE
// SIG // ChMOVmVyaVNpZ24sIEluYy4xNzA1BgNVBAsTLkNsYXNz
// SIG // IDMgUHVibGljIFByaW1hcnkgQ2VydGlmaWNhdGlvbiBB
// SIG // dXRob3JpdHkwHhcNMDQwNzE2MDAwMDAwWhcNMTQwNzE1
// SIG // MjM1OTU5WjCBtDELMAkGA1UEBhMCVVMxFzAVBgNVBAoT
// SIG // DlZlcmlTaWduLCBJbmMuMR8wHQYDVQQLExZWZXJpU2ln
// SIG // biBUcnVzdCBOZXR3b3JrMTswOQYDVQQLEzJUZXJtcyBv
// SIG // ZiB1c2UgYXQgaHR0cHM6Ly93d3cudmVyaXNpZ24uY29t
// SIG // L3JwYSAoYykwNDEuMCwGA1UEAxMlVmVyaVNpZ24gQ2xh
// SIG // c3MgMyBDb2RlIFNpZ25pbmcgMjAwNCBDQTCCASIwDQYJ
// SIG // KoZIhvcNAQEBBQADggEPADCCAQoCggEBAL687rx+74Pr
// SIG // 4DdP+wMQOL4I0ox9nfqSfxkMwmvuQlKM3tMcSBMl6sFj
// SIG // evlRZe7Tqjv18JScK/vyZtQk2vf1n24ZOTa80KN2CB4i
// SIG // JyRsOJEn4oRJrhuKof0lgiwQMOhxqyjod0pR8ezN+PBU
// SIG // 1G/A420Kj9nYZI1jsi1OJ/aFDv5t4ymZ4oVHfC2Gf+hX
// SIG // j61nwjMykRMg/KkjFJptwoRLdmgE1XEsXSH6iA0m/R8t
// SIG // kSvnAVVN8m01KILf2WtcttbZqoH9X82DumOd0CL8qTtC
// SIG // abKOOrW8tJ4PXsTqLIKLKP1TCJbdtQEg0fmlGOfA7lFw
// SIG // N+G2BUhSSG846sPobHtEhLsCAwEAAaOCAaAwggGcMBIG
// SIG // A1UdEwEB/wQIMAYBAf8CAQAwRAYDVR0gBD0wOzA5Bgtg
// SIG // hkgBhvhFAQcXAzAqMCgGCCsGAQUFBwIBFhxodHRwczov
// SIG // L3d3dy52ZXJpc2lnbi5jb20vcnBhMDEGA1UdHwQqMCgw
// SIG // JqAkoCKGIGh0dHA6Ly9jcmwudmVyaXNpZ24uY29tL3Bj
// SIG // YTMuY3JsMB0GA1UdJQQWMBQGCCsGAQUFBwMCBggrBgEF
// SIG // BQcDAzAOBgNVHQ8BAf8EBAMCAQYwEQYJYIZIAYb4QgEB
// SIG // BAQDAgABMCkGA1UdEQQiMCCkHjAcMRowGAYDVQQDExFD
// SIG // bGFzczNDQTIwNDgtMS00MzAdBgNVHQ4EFgQUCPVR6Pv+
// SIG // PT1kNnxoz1t4qN+5xTcwgYAGA1UdIwR5MHehY6RhMF8x
// SIG // CzAJBgNVBAYTAlVTMRcwFQYDVQQKEw5WZXJpU2lnbiwg
// SIG // SW5jLjE3MDUGA1UECxMuQ2xhc3MgMyBQdWJsaWMgUHJp
// SIG // bWFyeSBDZXJ0aWZpY2F0aW9uIEF1dGhvcml0eYIQcLrk
// SIG // HRDZKTS2OMp7A8y6vzANBgkqhkiG9w0BAQUFAAOBgQCu
// SIG // Ohe4SntV+mRV7ECk7UlBkJmcibyvLh3KeCP5HBkPf+to
// SIG // vDLZiDje3D/TibQ/sYKW8aRauu0uJtPefAFuAAoApAaS
// SIG // EUgJQPkcGHlnIyTgu9XhUK4b9Q7d4C6BzYCjbFJPkXVV
// SIG // iroi8tLqQXWIL2NVfR5UWpVZytk0gcBfXvZ6tTCCBQkw
// SIG // ggPxoAMCAQICEErej526uUojb1I0pA35Pn4wDQYJKoZI
// SIG // hvcNAQEFBQAwgbQxCzAJBgNVBAYTAlVTMRcwFQYDVQQK
// SIG // Ew5WZXJpU2lnbiwgSW5jLjEfMB0GA1UECxMWVmVyaVNp
// SIG // Z24gVHJ1c3QgTmV0d29yazE7MDkGA1UECxMyVGVybXMg
// SIG // b2YgdXNlIGF0IGh0dHBzOi8vd3d3LnZlcmlzaWduLmNv
// SIG // bS9ycGEgKGMpMDQxLjAsBgNVBAMTJVZlcmlTaWduIENs
// SIG // YXNzIDMgQ29kZSBTaWduaW5nIDIwMDQgQ0EwHhcNMDgw
// SIG // MzIwMDAwMDAwWhcNMDkwMzIwMjM1OTU5WjCBzDELMAkG
// SIG // A1UEBhMCVVMxEzARBgNVBAgTCkNhbGlmb3JuaWExEzAR
// SIG // BgNVBAcTClNhbiBSYWZhZWwxFzAVBgNVBAoUDkF1dG9k
// SIG // ZXNrLCBJbmMuMSEwHwYDVQQLFBhQU0VCLUxpY2Vuc2lu
// SIG // ZyAmIEluc3RhbGwxPjA8BgNVBAsTNURpZ2l0YWwgSUQg
// SIG // Q2xhc3MgMyAtIE1pY3Jvc29mdCBTb2Z0d2FyZSBWYWxp
// SIG // ZGF0aW9uIHYyMRcwFQYDVQQDFA5BdXRvZGVzaywgSW5j
// SIG // LjCBnzANBgkqhkiG9w0BAQEFAAOBjQAwgYkCgYEAp+6n
// SIG // nTaWIaaik4EgXpsaskB3OTSOClzOXYCXAb72SwJCpwl9
// SIG // vER/uhH7cq2P7d6UNPue0OFVpr73qgMo0/TVXms6jug1
// SIG // n+nHo55JZcMOaDDprfsuuDznAmsiMoZoAdnMO1n3ZBbp
// SIG // gXcZ6xujdhb+wllM1s165yyYM+tTZlI53s8CAwEAAaOC
// SIG // AX8wggF7MAkGA1UdEwQCMAAwDgYDVR0PAQH/BAQDAgeA
// SIG // MEAGA1UdHwQ5MDcwNaAzoDGGL2h0dHA6Ly9DU0MzLTIw
// SIG // MDQtY3JsLnZlcmlzaWduLmNvbS9DU0MzLTIwMDQuY3Js
// SIG // MEQGA1UdIAQ9MDswOQYLYIZIAYb4RQEHFwMwKjAoBggr
// SIG // BgEFBQcCARYcaHR0cHM6Ly93d3cudmVyaXNpZ24uY29t
// SIG // L3JwYTATBgNVHSUEDDAKBggrBgEFBQcDAzB1BggrBgEF
// SIG // BQcBAQRpMGcwJAYIKwYBBQUHMAGGGGh0dHA6Ly9vY3Nw
// SIG // LnZlcmlzaWduLmNvbTA/BggrBgEFBQcwAoYzaHR0cDov
// SIG // L0NTQzMtMjAwNC1haWEudmVyaXNpZ24uY29tL0NTQzMt
// SIG // MjAwNC1haWEuY2VyMB8GA1UdIwQYMBaAFAj1Uej7/j09
// SIG // ZDZ8aM9beKjfucU3MBEGCWCGSAGG+EIBAQQEAwIEEDAW
// SIG // BgorBgEEAYI3AgEbBAgwBgEBAAEB/zANBgkqhkiG9w0B
// SIG // AQUFAAOCAQEAoPt/JaNzwVvcuubHYt0M860+ntlMVoew
// SIG // XuvpZovRS8qzZTdLQDOGL4DL5GfVcC3t8YyomUYLJ7Rh
// SIG // BGhjuPyDtinUxw85rWZ+OHDkSvmPL0seDyJeUELj17qU
// SIG // JHCwJyeAErkw8vt7Tyi8XZjF2YP3PCGBYTX2s2qMTlXZ
// SIG // z6nqHnbN564AtxqBHcu4HXoKUT28fFhZEmYIANzmRaTi
// SIG // ZBYFFxaAf89H9xBSE+WqOiLP0W7FGJ6FTRyCIAkwFOfY
// SIG // Xi34nFfP62xKRrD7O2BPKI1mMlUSizqRFpooCWfjZjcw
// SIG // XWGQu6TZ9Xf+1DoLHrYbOPIjmpuTEyZtr8s94s/9eof6
// SIG // pDGCA6owggOmAgEBMIHJMIG0MQswCQYDVQQGEwJVUzEX
// SIG // MBUGA1UEChMOVmVyaVNpZ24sIEluYy4xHzAdBgNVBAsT
// SIG // FlZlcmlTaWduIFRydXN0IE5ldHdvcmsxOzA5BgNVBAsT
// SIG // MlRlcm1zIG9mIHVzZSBhdCBodHRwczovL3d3dy52ZXJp
// SIG // c2lnbi5jb20vcnBhIChjKTA0MS4wLAYDVQQDEyVWZXJp
// SIG // U2lnbiBDbGFzcyAzIENvZGUgU2lnbmluZyAyMDA0IENB
// SIG // AhBK3o+durlKI29SNKQN+T5+MAkGBSsOAwIaBQCggbQw
// SIG // GQYJKoZIhvcNAQkDMQwGCisGAQQBgjcCAQQwHAYKKwYB
// SIG // BAGCNwIBCzEOMAwGCisGAQQBgjcCARUwIwYJKoZIhvcN
// SIG // AQkEMRYEFOaosAnRZpLo8tH5m9cXzpAKfLwBMFQGCisG
// SIG // AQQBgjcCAQwxRjBEoCaAJABBAHUAdABvAGQAZQBzAGsA
// SIG // IABDAG8AbQBwAG8AbgBlAG4AdKEagBhodHRwOi8vd3d3
// SIG // LmF1dG9kZXNrLmNvbSAwDQYJKoZIhvcNAQEBBQAEgYBX
// SIG // tZtz+GXQWVDOkSukC6/xj7wvQ7lBF2DG9Uc7u3JYhSSA
// SIG // ByDAG4fCyws3v/9YfJ+Kp9ZkAIobfNHT1rnxyK49x0Ry
// SIG // /H2ML73MVhZbg/ViswZZ+xSQgyR8ruQsDxF4apdIi5Cp
// SIG // eRNDlcALNWYut8ap0zc54eprI5wuLtweU2CQxKGCAX8w
// SIG // ggF7BgkqhkiG9w0BCQYxggFsMIIBaAIBATBnMFMxCzAJ
// SIG // BgNVBAYTAlVTMRcwFQYDVQQKEw5WZXJpU2lnbiwgSW5j
// SIG // LjErMCkGA1UEAxMiVmVyaVNpZ24gVGltZSBTdGFtcGlu
// SIG // ZyBTZXJ2aWNlcyBDQQIQOCXX+vhhr570kOcmtdZa1TAJ
// SIG // BgUrDgMCGgUAoF0wGAYJKoZIhvcNAQkDMQsGCSqGSIb3
// SIG // DQEHATAcBgkqhkiG9w0BCQUxDxcNMDkwMjAzMjI1NTIz
// SIG // WjAjBgkqhkiG9w0BCQQxFgQUwowKLJytZT9VJeBFwCWC
// SIG // BrFwYM0wDQYJKoZIhvcNAQEBBQAEgYBpop/cgCHgQ3wp
// SIG // TsuHjn1Z3rlgQESQeEEKtEINIFU/dF7NljvdlWnDefWD
// SIG // 6xj7rxYwsnGUbioOExMdRYnwwst5bkkmqSzVMRLE7REU
// SIG // h7A/Ywi24MT4yx9UPkXU1W0BxN4ypr7Z1o0+egmwmx+B
// SIG // ryVjuAa4lEeBTqCHnYsYLAwXkw==
// SIG // End signature block
